<?php

namespace Hello2Forms\Service;

use Hello2Forms\Repository\WorkspaceRepository;
use HTMLPurifier;
use HTMLPurifier_Config;
use Illuminate\Support\Arr;
use function collect;

class FormCleanerService {
	/**
	 * All the performed cleanings
	 */
	private array $cleanings = [];

	private array $data;

	// For remove keys those have empty value
	private array $customKeys = [ 'seo_meta' ];

	private array $formDefaults = [
		'notifies'               => false,
		'no_branding'            => false,
		'webhook_url'            => null,
		'database_fields_update' => null,
		'slack_webhook_url'      => null,
		'discord_webhook_url'    => null,
		'editable_submissions'   => false,
		'custom_code'            => null,
		'seo_meta'               => []
	];

	private array $fieldDefaults = [
		'file_upload' => false,
	];

	// Escaped messages for each cleaning step in getPerformedCleanings function.
	private array $cleaningMessages = [
		// For form
		'notifies'               => "Email notification were disabled.",
		'no_branding'            => "Branding is not hidden.",
		'webhook_url'            => "Webhook disabled.",
		'database_fields_update' => 'Form submission will only create new records (no updates).',
		'slack_webhook_url'      => "Slack webhook disabled.",
		'discord_webhook_url'    => "Discord webhook disabled.",
		'editable_submissions'   => 'Users will not be able to edit their submissions.',
		'custom_code'            => 'Custom code was disabled',
		'seo_meta'               => 'Custom SEO was disabled',

		// For fields
		'file_upload'            => "Link field is not a file upload.",
		'custom_block'           => 'The custom block was removed.',
	];

	/**
	 * Returns form data after request ingestion
	 * @return array
	 */
	public function getData(): array {
		return $this->data;
	}

	/**
	 * Returns true if at least one cleaning was done
	 * @return bool
	 */
	public function hasCleaned(): bool {
		return count( $this->cleanings ) > 0;
	}

	/**
	 * Returns the messages for each cleaning step performed
	 */
	public function getPerformedCleanings(): array {
		$cleaningMsgs = [];
		foreach ( $this->cleanings as $key => $val ) {
			$cleaningMsgs[ $key ] = collect( $val )->map( function ( $cleaning ) {
				return esc_html__( $this->cleaningMessages[ $cleaning ], HF_TEXTDOMAIN );
			} );
		}

		return $cleaningMsgs;
	}

	/**
	 * Removes form pro features from data if user isn't pro
	 */
	public function processData( array $data ): self {
		$this->data = $this->commonCleaning( $data );

		return $this;
	}

	/**
	 * Create form cleaner instance from existing form
	 */
	public function processForm( array $form ): self {
		$this->data = $this->commonCleaning( $form );

		return $this;
	}

	private function isPro( int $workspace ): bool {
		$workspaceRepository = new WorkspaceRepository();
		return $workspaceRepository->isPro( $workspace );
	}

	/**
	 * Dry run celanings
	 *
	 * @param int $workspace
	 *
	 * @return FormCleanerService
	 */
	public function simulateCleaning( int $workspace ): self {
		if ( ! $this->isPro( $workspace ) ) {
			$this->data = $this->removeProFeatures( $this->data, true );
		}

		return $this;
	}

	/**
	 * Perform Cleanigns
	 *
	 * @param int $workspace
	 *
	 * @return FormCleanerService
	 */
	public function performCleaning( int $workspace ): self {
		if ( ! $this->isPro( $workspace ) ) {
			$this->data = $this->removeProFeatures( $this->data );
		}

		return $this;
	}

	/**
	 * Clean all forms:
	 * - Escape html of custom text block
	 */
	private function commonCleaning( array $data ): array {
		foreach ( $data['properties'] as &$property ) {
			if ( $property['type'] == 'nf-text' && isset( $property['content'] ) ) {
				$config = HTMLPurifier_Config::createDefault();
				$purifier = new HTMLPurifier($config);
				$property['content'] = $purifier->purify($property['content']);
			}
		}

		return $data;
	}

	private function removeProFeatures( array $data, $simulation = false ) {
		$this->cleanForm( $data, $simulation );
		$this->cleanProperties( $data, $simulation );

		return $data;
	}

	private function cleanForm( array &$data, $simulation = false ): void {
		$this->clean( $data, $this->formDefaults, $simulation );
	}

	private function cleanProperties( array &$data, $simulation = false ): void {
		foreach ( $data['properties'] as $key => &$property ) {
			$this->cleanField( $property, $this->fieldDefaults, $simulation );
		}
	}

	private function clean( array &$data, array $defaults, $simulation = false ): void {
		foreach ( $defaults as $key => $value ) {

			// Get value from form
			$formVal = Arr::get( $data, $key );

			// Transform customkeys values
			$formVal = $this->cleanCustomKeys( $key, $formVal );

			// Transform boolean values
			$formVal = ( ( $formVal === 0 || $formVal === "0" ) ? false : $formVal );
			$formVal = ( ( $formVal === 1 || $formVal === "1" ) ? true : $formVal );

			if ( ! is_null( $formVal ) && $formVal !== $value ) {
				if ( ! isset( $this->cleanings['form'] ) ) {
					$this->cleanings['form'] = [];
				}
				$this->cleanings['form'][] = $key;

				// If not a simulation, do the cleaning
				if ( ! $simulation ) {
					Arr::set( $data, $key, $value );
				}
			}
		}
	}

	private function cleanField( array &$data, array $defaults, $simulation = false ): void {
		foreach ( $defaults as $key => $value ) {
			if ( isset( $data[ $key ] ) && Arr::get( $data, $key ) !== $value ) {
				$this->cleanings[ $data['name'] ][] = $key;
				if ( ! $simulation ) {
					Arr::set( $data, $key, $value );
				}
			}
		}
	}

	// Remove keys those have empty value
	private function cleanCustomKeys( $key, $formVal ) {
		if ( in_array( $key, $this->customKeys ) && $formVal !== null ) {
			$newVal = [];
			foreach ( $formVal as $k => $val ) {
				if ( $val ) {
					$newVal[ $k ] = $val;
				}
			}

			return $newVal;
		}

		return $formVal;
	}

}
