<?php

namespace Hello2Forms\Controller;

use Hello2Forms\Architecture\AbstractApiController;
use Hello2Forms\Architecture\Integration;
use Hello2Forms\Repository\IntegrationAccountsRepository;

class IntegrationAccountsController extends AbstractApiController {

	// Register our routes.
	public function registerRoutes(): void {

		$this->registerRestRoute( '', array(

			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'store' ),
				'permission_callback' => function () {
					return current_user_can( 'manage_hello2forms' );
				},
			),
		) );

		$this->registerRestRoute( '(?P<type>[\w\.-]+)', array(

			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'list' ),
				'permission_callback' => function () {
					return current_user_can( 'manage_hello2forms' );
				},
			),
		) );
	}


	function list( \WP_REST_Request $request ) {
		$integrationAccountsRepository = new IntegrationAccountsRepository();
		$type                          = $request->get_param( 'type' );

		$accounts                      = $integrationAccountsRepository->findByUserIdAndType( get_current_user_id(), $type);

		if ( ! $accounts ) {
			return rest_ensure_response( new \WP_REST_Response( [
				'type' => 'success',
				'data' => [],
			], 201 ));
		}

		return rest_ensure_response( new \WP_REST_Response( [
			'type' => 'success',
			'data' => array_map( function( $account ) {
				return [
					'value' => $account['id'],
					'name' => $account['name'],
				];
			}, $accounts ),
		], 201 ));
	}

	function store( \WP_REST_Request $request ) {
		$integrationAccountsRepository = new IntegrationAccountsRepository();
		$type                          = $request->get_param( 'type' );

		if ( ! in_array( $type, Integration::LIST ) ) {
			return new \WP_REST_Response(
				[
					'error' => 'Invalid type.',
				],
				400 );
		}

		$account =  $request->get_param( 'account' );

		if ( count( array_diff($account, Integration::STRUCTURES[ $type ] ) ) === 0 ) {
			return new \WP_REST_Response(
				[
					'error' => esc_attr__('Invalid data structure', HF_TEXTDOMAIN),
				],
				400 );
		}

		$integrationAccountsRepository->store(
			$request->get_param( 'name' ),
			$type,
			$account,
			get_current_user_id()
		);

		// Return all of our comment response data.
		return rest_ensure_response( new \WP_REST_Response( [
			'type' => 'success',
			'message' => esc_html__('Integration account created successfully', HF_TEXTDOMAIN)
		], 201 ));
	}

}
